<?php
// register.php
require_once __DIR__ . '/common.php'; // gives $pdo, json_out(), read_json(), is_lk(), normalize_sl_phone()

// Toggle extra debug via query param: /register.php?debug=1 (only for development)
$APP_DEBUG = isset($_GET['debug']) && $_GET['debug'] == '1';

if (!headers_sent()) {
  header('Content-Type: application/json; charset=utf-8');
}

$step = 'start';

try {
  $in = read_json();
  $name       = trim((string)($in['name'] ?? ''));
  $email      = strtolower(trim((string)($in['email'] ?? '')));
  $phone_in   = trim((string)($in['phone'] ?? ''));
  $country_in = strtoupper(trim((string)($in['country'] ?? 'LK'))); // e.g. 'LK'
  $password   = (string)($in['password'] ?? '');

  // --- validations ---
  $step = 'validate';

  if ($email === '' || !filter_var($email, FILTER_VALIDATE_EMAIL)) {
    json_out(['ok'=>false, 'error'=>'Valid email is required']);
  }
  if ($password === '' || strlen($password) < 6) {
    json_out(['ok'=>false, 'error'=>'Password too short (min 6)']);
  }
  if ($name !== '' && mb_strlen($name) > 120) {
    json_out(['ok'=>false, 'error'=>'Name is too long']);
  }

  // Phone is required for LK; store SL numbers normalized as +94xxxxxxxxx; others optional.
  $phone = null;
  if (is_lk($country_in)) {
    if ($phone_in === '') {
      json_out(['ok'=>false, 'error'=>'Phone is required for Sri Lanka']);
    }
    $phone = normalize_sl_phone($phone_in); // -> +94xxxxxxxxx
  } else {
    // Optional for non-LK. You can still save as provided if you want.
    $phone = $phone_in !== '' ? $phone_in : null;
  }

  // --- duplicate checks ---
  $step = 'dups';
  $emailExists = false;
  $phoneExists = false;

  $q = $pdo->prepare("SELECT 1 FROM users WHERE LOWER(email) = ? LIMIT 1");
  $q->execute([$email]);
  $emailExists = (bool)$q->fetchColumn();

  if ($phone !== null) {
    $q2 = $pdo->prepare("SELECT 1 FROM users WHERE phone = ? LIMIT 1");
    $q2->execute([$phone]);
    $phoneExists = (bool)$q2->fetchColumn();
  }

  if ($emailExists || $phoneExists) {
    json_out([
      'ok'            => false,
      'error'         => $emailExists ? 'This email already has an account' : 'This phone already has an account',
      'email_exists'  => $emailExists,
      'phone_exists'  => $phoneExists,
    ]);
  }

  // --- insert user (OTP removed entirely) ---
  $step = 'insert';
  $pdo->beginTransaction();

  $hash = password_hash($password, PASSWORD_BCRYPT);

  // users table columns:
  // id, name, email, phone, country_code, password_hash,
  // phone_verified_at, email_verified_at, is_active, created_at, updated_at
  $ins = $pdo->prepare("
    INSERT INTO users
      (name, email, phone, country_code, password_hash, phone_verified_at, email_verified_at, is_active, created_at, updated_at)
    VALUES
      (?,    ?,     ?,     ?,            ?,             NULL,              NULL,              1,         NOW(),     NOW())
  ");
  $ins->execute([
    $name ?: null,
    $email,
    $phone,        // null for non-LK without phone
    $country_in,   // country_code
    $hash
  ]);

  $userId = (int)$pdo->lastInsertId();

  $pdo->commit();

  // No OTP creation, no SMS sending. User is ready right away.
  json_out(['ok' => true, 'user_id' => $userId]);

} catch (Throwable $e) {
  if ($pdo && $pdo->inTransaction()) {
    $pdo->rollBack();
  }
  error_log('register.php error ['.$step.']: '.$e->getMessage());
  $out = ['ok'=>false, 'error'=>'Server error'];
  if ($APP_DEBUG) {
    $out['debug'] = ['step'=>$step, 'message'=>$e->getMessage()];
  }
  json_out($out);
}
