<?php
// compute_delivery_fee.php
require_once __DIR__ . '/common.php';

// IMPORTANT: put your Google Directions key into env GOOGLE_MAPS_KEY
define('GMAPS_KEY', getenv('GOOGLE_MAPS_KEY') ?: 'AIzaSyB2-q_0T2LB2sA1X_syazc4Y-yZ-IdmSvo');

function json_bad($msg){ json_out(['ok'=>false,'error'=>$msg]); }

$in = read_json();
$store_id = (int)($in['store_id'] ?? 0);
$dest_lat = (float)($in['dest_lat'] ?? 0);
$dest_lng = (float)($in['dest_lng'] ?? 0);

if ($store_id <= 0) json_bad('store_id required');
if (!$dest_lat || !$dest_lng) json_bad('Destination lat/lng required');

try {
  // 1) Fetch store coords
  $q = $pdo->prepare("SELECT id, name, lat, lng FROM stores WHERE id = ? LIMIT 1");
  $q->execute([$store_id]);
  $store = $q->fetch();
  if (!$store || !$store['lat'] || !$store['lng']) json_bad('Store location not set');

  $oLat = (float)$store['lat']; $oLng = (float)$store['lng'];
  $dLat = $dest_lat;            $dLng = $dest_lng;

  // 2) Driving distance (Google Directions); fallback to Haversine
  $source = 'google';
  $km = 0.0;

  if (GMAPS_KEY) {
    $km = driving_distance_km_google($oLat,$oLng,$dLat,$dLng, GMAPS_KEY);
  } else {
    $km = 0.0;
  }

  if ($km <= 0.0) {
    $km = haversine_km($oLat,$oLng,$dLat,$dLng);
    $source = 'haversine';
  }

  // 3) Find pricing slab
  // Prefer store-specific; fallback to global
  $slab = find_pricing_slab($pdo, $store_id, $km);
  if (!$slab) $slab = find_pricing_slab($pdo, null, $km);
  if (!$slab) json_bad('No pricing slab configured');

  $fee = compute_fee_from_slab($slab, $km);

  json_out([
    'ok' => true,
    'distance_km' => round($km, 3),
    'fee' => round($fee),
    'pricing' => [
      'slab_id' => (int)$slab['id'],
      'base_fee'=> (float)$slab['base_fee'],
      'per_km'  => (float)$slab['per_km'],
      'free_km' => (float)$slab['free_km'],
      'min_fee' => (float)$slab['min_fee'],
    ],
    'source' => $source
  ]);

} catch (Throwable $e) {
  error_log('compute_delivery_fee error: '.$e->getMessage());
  json_bad('Server error');
}

// ---------- helpers ----------

function driving_distance_km_google($oLat,$oLng,$dLat,$dLng,$key){
  $url = 'https://maps.googleapis.com/maps/api/directions/json?' . http_build_query([
    'origin' => "{$oLat},{$oLng}",
    'destination' => "{$dLat},{$dLng}",
    'mode' => 'driving',
    'units' => 'metric',
    'key' => $key
  ]);

  $ch = curl_init($url);
  curl_setopt_array($ch, [
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_TIMEOUT => 15,
  ]);
  $body = curl_exec($ch);
  $err  = curl_error($ch);
  curl_close($ch);

  if ($err || !$body) return 0.0;

  $j = json_decode($body, true);
  if (!is_array($j) || ($j['status'] ?? '') !== 'OK') return 0.0;

  // Sum all legs distance.value (meters)
  $meters = 0;
  foreach (($j['routes'][0]['legs'] ?? []) as $leg) {
    $meters += (int)($leg['distance']['value'] ?? 0);
  }
  return $meters > 0 ? ($meters / 1000.0) : 0.0;
}

function haversine_km($lat1,$lon1,$lat2,$lon2){
  $R = 6371; // km
  $dLat = deg2rad($lat2-$lat1);
  $dLon = deg2rad($lon2-$lon1);
  $a = sin($dLat/2)*sin($dLat/2) + cos(deg2rad($lat1))*cos(deg2rad($lat2))*sin($dLon/2)*sin($dLon/2);
  $c = 2 * atan2(sqrt($a), sqrt(1-$a));
  return $R * $c;
}

function find_pricing_slab(PDO $pdo, $store_id_or_null, $km){
  if ($store_id_or_null === null) {
    $q = $pdo->prepare("
      SELECT * FROM delivery_pricing
      WHERE store_id IS NULL
        AND min_km <= ?
        AND (max_km IS NULL OR ? <= max_km)
      ORDER BY min_km DESC
      LIMIT 1
    ");
    $q->execute([$km, $km]);
    return $q->fetch();
  } else {
    $q = $pdo->prepare("
      SELECT * FROM delivery_pricing
      WHERE store_id = ?
        AND min_km <= ?
        AND (max_km IS NULL OR ? <= max_km)
      ORDER BY min_km DESC
      LIMIT 1
    ");
    $q->execute([$store_id_or_null, $km, $km]);
    return $q->fetch();
  }
}

function compute_fee_from_slab($slab, $km){
  $base = (float)$slab['base_fee'];
  $per  = (float)$slab['per_km'];
  $free = (float)$slab['free_km'];
  $minf = (float)$slab['min_fee'];

  $billable_km = max(0.0, $km - $free);
  $fee = $base + ($per * $billable_km);
  if ($minf > 0) $fee = max($fee, $minf);
  return $fee;
}
